<?php

App::uses('BuzzBookingsAppModel', 'BuzzBookings.Model');

class Extra extends BuzzBookingsAppModel {

	/**
	 * Has many associations
	 *
	 * @var array
	 */
	public $hasMany = array(
		'ExtrasUpsellExclusions' => array(
			'className' => 'BuzzBookings.ExtrasUpsellExclusions'
		)
	);

	/**
	 * HABTM associations
	 *
	 * @var array
	 */
	public $hasAndBelongsToMany = array(
		'UpsellApplicableActivity' => array(
			'className' => 'BuzzBookings.Activity',
			'joinTable' => 'extras_upsell_applicable_activities'
		),
		'UpsellExclusion' => array(
			'className' => 'BuzzBookings.ActivityPackage',
			'joinTable' => 'extras_upsell_exclusions'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'validate.maxLength150'
			)
		),
		'api_reference' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'format' => array(
				'rule' => '|^#\w+$|',
				'message' => 'Invalid API reference'
			)
		),
		'price' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'decimal' => array(
				'rule' => '|^\d+(\.\d{2})?$|',
				'message' => 'Decimal numbers only'
			)
		),
		'sequence' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('naturalNumber', true)
			)
		)
	);

	/**
	 * Image slots
	 *
	 * @var array
	 */
	public $imageSlots = [
		'main' => [
			'slots' => 1,
			'fields' => []
		]
	];

	/**
	 * Read for edit
	 *
	 * @param int $id
	 * @param array $query
	 * @return array
	 */
	public function readForEdit($id, $query = []) {
		$query['contain'][] = 'UpsellApplicableActivity';
		$query['contain'][] = 'UpsellExclusion';
		return parent::readForEdit($id, $query);
	}

	/**
	 * Returns extras for the specified activity type and packages.
	 *
	 * @param int $activityId
	 * @param array $packages
	 * @param int $limit
	 * @return array
	 */
	public function getAvailableExtras($activityId, array $packages, $limit = 3) {
		// Get ALL extras for the activity.
		$data = $this->find(
			'all',
			array(
				'contain' => [
					'Image'
				],
				'joins' => array(
					array(
						'table' => 'extras_upsell_applicable_activities',
						'alias' => 'ExtrasUpsellApplicableActivities',
						'type' => 'INNER',
						'conditions' => array(
							'Extra.id = ExtrasUpsellApplicableActivities.extra_id',
							'ExtrasUpsellApplicableActivities.activity_id' => $activityId
						)
					)
				),
				'conditions' => array(
					'Extra.is_active' => true,
					'Extra.is_removed' => false
				),
				'order' => array(
					'Extra.sequence' => 'ASC'
				)
			)
		);

		// Get the packages excluded from these extras.
		$excludeExtras = $this->ExtrasUpsellExclusions->find(
			'list',
			array(
				'fields' => array(
					'ExtrasUpsellExclusions.id',
					'ExtrasUpsellExclusions.extra_id'
				),
				'conditions' => array(
					'ExtrasUpsellExclusions.extra_id' => Hash::extract($data, '{n}.Extra.id'),
					'ExtrasUpsellExclusions.activity_package_id' => $packages
				)
			)
		);

		// Remove extras excluded from packages.
		$data = array_filter($data, function ($val) use ($excludeExtras) {
			return !in_array($val['Extra']['id'], $excludeExtras);
		});

		// Return the maximum number of found extras.
		return array_splice($data, 0, $limit);
	}

}
