<?php
App::uses('BuzzBlogAppModel', 'BuzzBlog.Model');

class BlogPost extends BuzzBlogAppModel {

	public $actsAs = array(
		'Routable.Routable' => array(
			'alias' => 'blog/:displayField',
			'config' => 'BuzzBlog'
		),
		'MetaData.Meta',
	);

	public $displayField = 'name';

	public $imageSlots = array(
		'listing' => array(
			'slots' => 1,
			'fields' => array(
				'alt'
			)
		),
		'main' => array(
			'slots' => -1,
			'fields' => array(
				'alt'
			)
		),
	);

	public $hasAndBelongsToMany = array(
		'BlogCategory' => array(
			'className' => 'BuzzBlog.BlogCategory',
			'joinTable' => 'buzz_blog_blog_categories_blog_posts'
		)
	);

	public $hasMany = array(
		'BlogTag' => array(
			'className' => 'BuzzBlog.BlogTag',
			'cascade' => true,
			'dependent' => true
		),
		'BlogComment' => array(
			'className' => 'BuzzBlog.BlogComment',
			'cascade' => true,
			'dependent' => true
		)
	);

	public $order = array(
		'BlogPost.posted' => 'DESC'
	);

	public $validate = array(
		'name' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please add a name'
			),
		),
		'posted' => array(
			'datetime' => array(
				'rule' => array('datetime'),
				'message' => 'Please add a posted date and time'
			),
		),
		'body' => array(
			'notBlank' => array(
				'rule' => array('notBlank'),
				'message' => 'Please add body text'
			),
		),
	);

	/**
	 * Constructor
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		parent::__construct($id, $table, $ds);

		// If BuzzGallery loaded associate activities with galleries. We're
		// handling this in the constructor so that this feature can easily
		// be disabled on a site if BuzzGallery is excluded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->belongsTo['Gallery'] = array(
				'className' => 'BuzzGallery.Gallery'
			);
		}
	}

	public function beforeValidate($options = array()) {
		// Force a category to be selected.
		if (isset($this->data['BlogCategory']['BlogCategory']) && empty($this->data['BlogCategory']['BlogCategory'])) {
			$this->BlogCategory->invalidate('BlogCategory', 'Please select at least one category to publish this post under.');
		}
		return parent::beforeValidate($options);
	}

/**
 * Executes query based on supplied params and returns the found data
 *
 * @param int $id The unique id of the requested item
 * @param array $query Contains additional parameters to insert into the find call
 * @return array Containins the resulting data
 */
	public function readForView($id, $query = array()) {
		$query['contain'][] = 'BlogTag';

		$query['contain']['BlogComment'] = array(
			'conditions' => array(
				'BlogComment.is_active' => 1
			),
			'order' => 'BlogComment.created ASC',
		);

		$query['contain'][] = 'BlogCategory';

		return parent::readForView($id, $query);
	}

	public function readForEdit($id, $query = array()) {
		$query['contain'][] = 'BlogTag';
		$query['contain']['BlogComment'] = array(
			'order' => 'BlogComment.created ASC'
		);
		$query['contain'][] = 'BlogCategory';

		return parent::readForEdit($id, $query);
	}

	public function beforeSave($options = array()) {
		if (isset($this->data['BlogPost']['blog_category_id']) && $this->data['BlogPost']['blog_category_id'] == 0) {
			$this->data['BlogPost']['blog_category_id'] = null;
		}

		if (!empty($this->data['BlogPost']['posted']) && !empty($this->data['BlogPost']['posted_gmt'])) {
			// Convert date to GMT.
			$this->data['BlogPost']['posted'] = CakeTime::toServer($this->data['BlogPost']['posted'], Configure::read('Config.timezone'));
		}

		return parent::beforeSave($options);
	}

	public function afterFind($results, $primary = false) {
		$results = parent::afterFind($results, $primary);
		$length = Configure::read('BuzzBlog.snippet_length');

		foreach ($results as $k => $blog) {
			if (isset($blog['BlogPost'])) {

				if (isset($blog['BlogPost']['body']) && !empty($blog['BlogPost']['body'])) {
					$snippet = strip_tags($blog['BlogPost']['body']);

					if (strlen($snippet) > $length) {
						$snippet = substr($snippet, 0, $length) . ' &hellip;';
					}

					$results[$k]['BlogPost']['snippet'] = $snippet;
				}
			}

			if (! isset($blog['BlogPost']['blog_category_id']) || empty($blog['BlogPost']['blog_category_id'])) {
				$results[$k]['BlogPost']['blog_category_id'] = 0;
			}

			// implode all the tags into one string
			if (isset($blog['BlogTag']) && ! empty($blog['BlogTag'])) {
				// sort the tags
				$tags = array();
				foreach ($blog['BlogTag'] as $tag) {
					$tags[] = $tag['tag'];
				}

				$results[$k]['BlogPost']['tags'] = implode(', ', $tags);

			} else {
				$results[$k]['BlogPost']['tags'] = '';
			}
		}

		return $results;
	}

	public function afterSave($created, $options = array()) {
		$this->BlogTag = ClassRegistry::init('BuzzBlog.BlogTag');

		if ($created === false) {
			$this->BlogTag->deleteAll(array(
				'BlogTag.blog_post_id' => $this->id
			));
		}

		if (isset($this->data['BlogPost']['tags']) && ! empty($this->data['BlogPost']['tags'])) {
			$tags = explode(',', $this->data['BlogPost']['tags']);
			$blogTags = array();

			foreach ($tags as $tag) {

				$blogTags[] = array(
					'BlogTag' => array(
						'tag' => trim($tag),
						'blog_post_id' => $this->id
					)
				);
			}

			unset($this->data['BlogPost']['tags']);
			$this->BlogTag->saveMany($blogTags);
		}

		return parent::afterSave($created, $options);
	}

}
