<?php
App::uses('BuzzBlogAppController', 'BuzzBlog.Controller');
App::uses('CakeTime', 'Utility');

class BlogPostsController extends BuzzBlogAppController {

	public $components = array('RequestHandler');

	public function beforeFilter() {
		parent::beforeFilter();

		$this->Auth->allow(array(
			'index',
			'view',
			'archives'
		));
	}

	protected function _adminFilterFields() {
		$filterFields = array();

		$Model = $this->{$this->modelClass};

		$filerFields[$Model->alias . '.name'] = array(
			'label' => 'Name',
			'type' => 'string',
			'compare' => array(
				$Model->alias . '.name LIKE' => '%%%s%%'
			)
		);

		// include an author filter field when authors are enabled in the config
		if (! isset($this->disablePostAuthor) || $this->disablePostAuthor === false) {
			$filerFields[$Model->alias . '.author'] = array(
				'label' => 'Author',
				'type' => 'string',
				'compare' => array(
					$Model->alias . '.author LIKE' => '%%%s%%'
				)
			);
		}

		$filerFields[$Model->alias . '.posted'] = array(
			'label' => 'Posted',
			'type' => 'date',
			'compare' => array($Model->alias . ".posted" => "%s")
		);

		return $filterFields;
	}

/**
 * override the admin index columns
 */
	protected function _adminIndexColumns() {
		$columns = parent::_adminIndexColumns();

		$Model = $this->{$this->modelClass};

		// remove the author column when requested to from the config
		if (
			(isset($this->disablePostAuthor) && $this->disablePostAuthor === true) &&
			! empty($columns[$Model->alias . '.author'])
		) {
			unset($columns[$Model->alias . '.author']);
		}

		// free up some space in the index by removing the created column
		unset($columns[$Model->alias . '.created']);

		$additionalColumns = array();

		// add concatinated blog category column
		if (empty($this->disableCategories)) {
			$additionalColumns[$Model->alias . '.blog_category_name_concat'] = array(
				'type' => 'string_full',
				'null' => true,
				'default' => null,
				'length' => (int)100,
				'collate' => 'utf8_general_ci',
				'charset' => 'utf8',
				'label' => 'Categories'
			);
		}

		// add posted column
		$additionalColumns[$Model->alias . '.posted'] = array(
			'type' => 'datetime',
			'null' => true,
			'default' => null,
			'length' => null,
			'label' => 'Posted On'
		);

		$addAdditionalColumnsAfter = isset($columns[$Model->alias . '.author']) ?
			$Model->alias . '.author':
			$Model->alias . '.name';

		return ArrayUtil::addAfter($columns, $addAdditionalColumnsAfter, $additionalColumns);
	}

	protected function _adminIndexPaginate() {
		$conditions = $this->_processFilter();

		$paginate = array(
			'conditions' => $conditions,
			'contain' => array(
				'BlogCategory'
			)
		);

		return $paginate;
	}

	protected function _adminFormFields() {
		$fields = parent::_adminFormFields();

		$Model = $this->{$this->modelClass};

		$additionalFields = array();

		// move the author field when disabled through the config
		if (isset($this->disablePostAuthor) && $this->disablePostAuthor === true) {
			unset($fields[$Model->alias . '.author']);
		}

		// add blog category multicheck field
		if (empty($this->disableCategories)) {
			$additionalFields['BlogCategory.BlogCategory'] = array(
				'type' => 'multicheck',
				'label' => 'Category'
			);
		}

		// we only want to show the tags elements when tags are enable in the config
		if (! isset($this->disableTags) || $this->disableTags === false) {
			$additionalFields[$Model->alias . '.tags'] = array(
				'type' => 'string',
				'after' => '&nbsp;&nbsp;Seperate multiple tags with a comma.'
			);
		}
		if (CakePlugin::loaded('BuzzGallery') !== true) {
			unset($fields['BlogPost.gallery_id']);
		}
		$fields['BlogPost.video_url']['label'] = 'Video URL';

		return ArrayUtil::addAfter($fields, $Model->alias . '.body', $additionalFields);
	}

/**
 * Admin populate lookups
 * @return void
 */
	protected function _adminPopulateLookups() {
		$BlogPost = $this->{$this->modelClass};
		$blogCategories = $BlogPost->BlogCategory->find('list');
		$this->set('blogCategories', $blogCategories);
		// Get galleries if the plugin is loaded.
		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->set('galleries', $BlogPost->Gallery->find('list'));
		}

		return;
	}

/**
 * Manages a default admin index view of this model
 * Includes a filter form and paginated results
 *
 * Override in your controller to customise
 *
 * @return void
 */
	public function admin_index() {
		$Model = $this->{$this->modelClass};
		$modelAlias = $Model->alias;

		parent::admin_index();

		// re-fetch paginate data so we can handle the displaying of concatinated blog categories
		$data = $this->paginate();

		// concatinate blog category names for each item to show on th admin index
		if (! empty($data)) {
			foreach ($data as $key => $item) {
				if (! empty($item['BlogCategory'])) {
					$blogCategories = Hash::combine($item, 'BlogCategory.{n}.id', 'BlogCategory.{n}.name');
					$data[$key][$modelAlias]['blog_category_name_concat'] = implode($blogCategories, ', ');
				}
			}
		}

		$this->set(compact('data'));

		return;
	}

/**
 * Manages an admin edit form for this model
 *
 * @param integer $id ID of row to edit
 * @return void
 */
	public function admin_edit($id = null) {
		parent::admin_edit($id);
		$this->view = 'BuzzBlog.BlogPosts/admin_edit';
	}

/**
 * blog index
 *
 * display blog listing page
 */
	public function index() {
		$Model = $this->{$this->modelClass};

		$this->loadModel('Page');

		// See if there is a linked content page
		$pageId = Configure::read('BuzzBlog.page_id');
		if (! empty($pageId)) {
			$this->assignPage($pageId, 'page');
		} else {
			$this->set('page', [
				'Page' => [
					'title' => __d('buzz_blog', 'Blog')
				]
			]);
		}

		if ($this->RequestHandler->isRss()) {
			$posts = $Model->find('all', array(
				'conditions' => array(
					'BlogPost.is_active' => 1,
					'BlogPost.posted <' => date('Y-m-d H:i:s')
				),
				'order' => 'BlogPost.posted DESC',
				'limit' => Configure::read('BuzzBlog.rss_limit')
			));

			$description = ! empty($page['Page']['body']) ? strip_tags($page['Page']['body']) : null;

			$descLen = strlen($description);

			if ($descLen > 0 && $descLen >= 250) {
				$description = String::truncate($description, 250);
			} elseif ($descLen == 0) {
				$description = ! empty($page['Page']['title']) ? strip_tags($page['Page']['title']) : null;
			}

			$channel = array();

			if (! empty($page)) {
				$channel = array(
					'title' => $page['Page']['title'],
					'link' => array(
						'plugin' => 'buzz_blog',
						'controller' => 'blog_posts',
						'action' => 'index'
					),
					'description' => $description,
					'language' => 'en-us'
				);
			}

			return $this->set(compact('channel', 'posts'));
		}

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					),
					'order' => 'BlogComment.created ASC'
				),
				'BlogCategory'
			),
			'limit' => Configure::read('BuzzBlog.listing_limit')
		);

		$this->set('data', $this->paginate());

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		$this->view = 'BuzzBlog.BlogPosts/index';
	}

/**
 * view individual blog post
 */
	public function view($id) {
		$BlogPost = $this->{$this->modelClass};
		$this->loadModel('Page');
		// See if there is a linked content page
		$pageId = Configure::read('BuzzBlog.page_id');
		if (! empty($pageId)) {
			$this->assignPage($pageId, 'page');
		} else {
			$this->set('page', [
				'Page' => [
					'title' => __d('buzz_blog', 'Blog')
				]
			]);
		}
		$data = $BlogPost->readForView($id);
		$this->Meta->set($data);
		$this->set('data', $data);
		// Call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		if (CakePlugin::loaded('BuzzGallery') === true) {
			$this->set('gallery', $BlogPost->Gallery->readForView($data['BlogPost']['gallery_id']));
		}

		$this->view = 'BuzzBlog.BlogPosts/view';

		return;
	}

/**
 * Blog archive, listing for blogs from a certian month
 *
 * @param string $date given month to get posts from
 */
	public function archives($date = false) {
		// Fall back incase no page is found
		$this->set('title_for_layout', "Blog");

		if (! $date) {
			$date = CakeTime::i18nFormat(time(), '%y-%m-%d');
		}

		$dateBits = explode('-', $date);

		$dateShort = $dateBits['0'] . '-' . $dateBits['1'];

		$category = array(
			'BlogArchive' => array(
				'name' => __d('buzz_blog', 'Blog Archive: %s', CakeTime::i18nFormat($date, '%B %Y'))
			)
		);

		$this->Meta->set($category, 'BlogArchive');
		$this->set('category', $category);

		$this->paginate = array(
			'conditions' => array(
				'BlogPost.is_active' => 1,
				'BlogPost.posted >=' => $dateShort . '-01 00:00:00',
				'BlogPost.posted <=' => $date . ' 23:59:59',
				'BlogPost.posted <' => date('Y-m-d H:i:s')
			),
			'order' => 'BlogPost.posted DESC',
			'contain' => array(
				'ListingImage',
				'BlogComment' => array(
					'conditions' => array(
						'BlogComment.is_active' => 1
					)
				),
				'BlogCategory'
			),
			'limit' => 10
		);

		$this->loadModel('BuzzBlog.BlogPost');
		$this->set('data', $this->paginate('BlogPost'));

		// call the widgets
		$this->widgetLatestPosts();
		$this->widgetCategories();
		$this->widgetArchives();

		$this->view = 'BuzzBlog.BlogPosts/archives';
	}

}
