<?php

App::uses('BuzzAdvertsAppModel', 'BuzzAdverts.Model');

class Advert extends BuzzAdvertsAppModel {

	public $findMethods = array('global' => true);

	/**
	 * Behaviors
	 *
	 * @var array
	 */
	public $actsAs = array(
		'BuzzTranslate.Translatable'
	);

	/**
	 * Belongs to assocations
	 *
	 * @var array
	 */
	public $belongsTo = array(
		'AdvertTemplate' => array(
			'className' => 'BuzzAdverts.AdvertTemplate'
		),
		'Theme' => array(
			'className' => 'BuzzTheme.Theme'
		)
	);

	/**
	 * Has one associations
	 *
	 * @var array
	 */
	public $hasOne = array(
		'AdvertRelation' => array(
			'className' => 'BuzzAdverts.AdvertRelation'
		)
	);

	/**
	 * Validation rules
	 *
	 * @var array
	 */
	public $validate = array(
		'name' => array(
			'required' => array(
				'rule' => 'notEmpty',
				'message' => 'Required'
			),
			'max' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters'
			)
		),
		'heading_text' => array(
			'max' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters',
				'allowEmpty' => true
			)
		),
		'subheading_text' => array(
			'max' => array(
				'rule' => array('maxLength', 150),
				'message' => 'No more than 150 characters',
				'allowEmpty' => true
			)
		),
		'link_text' => array(
			'max' => array(
				'rule' => array('maxLength', 45),
				'message' => 'No more than 45 characters',
				'allowEmpty' => true
			)
		),
		'url' => array(
			'url' => array(
				'rule' => 'validateUrls',
				'message' => 'Invalid URL',
				'allowEmpty' => true
			),
			'max' => array(
				'rule' => array('maxLength', 250),
				'message' => 'No more than 250 characters'
			)
		)
	);

	/**
	 * Constructor
	 */
	public function __construct($id = false, $table = null, $ds = null) {
		$this->imageSlots = array(
			'main' => array(
				'fields' => array(
					'alt' => array(
						'label' => __d('buzz_adverts', 'Alt Text'),
					)
				),
				'slots' => 1
			)
		);
		parent::__construct($id, $table, $ds);
	}

	/**
	 * Custom validation method for checking a URL (internal or external).
	 *
	 * @param array $data
	 * @return bool
	 */
	public function validateUrls(array $data) {
		$check = array_pop($data);

		$validates = preg_match('|^\/[\w\-_\/]+|', $check) === 1;
		if ($validates === false) {
			$validates = Validation::url($check);
		}

		return $validates;
	}

	/**
	 * After find.
	 *
	 * @param array $results
	 * @param bool $primary
	 * @return array
	 */
	public function afterFind($results, $primary = false) {
		$templates = [];

		// Check the adverts being loaded and get any required variables (this
		// is to handle special cases like Buy Gift Vouchers).
		if ($primary === false) {
			foreach ($results as $key => $val) {
				if (!empty($val['Advert']['view_template'])) {
					list($plugin, $template) = pluginSplit($val['Advert']['view_template']);
					$templates[$key] = $template;
				}
			}

			if (false !== $key = array_search('buy_gift_vouchers', $templates)) {
				$voucherActivities = ClassRegistry::init('BuzzVouchers.VoucherActivity')->listListed();
				$results[$key] = Hash::insert($results[$key], 'Advert.vars.voucherActivities', $voucherActivities);
			} elseif (false !== $key = array_search('facilities', $templates)) {
				$facilities = ClassRegistry::init('BuzzFacilities.Facility')->find(
					'all',
					[
						'contain' => ['ListingImage'],
						'conditions' => [
							'Facility.is_active' => true,
							'Facility.listed' => true
						]
					]
				);
				$results[$key] = Hash::insert($results[$key], 'Advert.vars.facilities', $facilities);
			}
		}

		return $results;
	}

	protected function _findGlobal($state, $query, $results = []) {
		if ($state === 'before') {
			$query['conditions']['Advert.is_global'] = true;
			return $query;
		}
		return $results;
	}

}
